<?php
session_start();
require __DIR__ . '/config.php';

// Protect page
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$error   = '';
$success = '';

// Optional: default prizes if table is empty (first time only)
$check = $pdo->query("SELECT COUNT(*) AS c FROM win_probabilities");
$count = (int)$check->fetch()['c'];
if ($count === 0) {
    $defaultPrizes = [1, 2, 5, 10, 20, 50];
    $insert = $pdo->prepare("
        INSERT INTO win_probabilities (prize_amount, weight, is_active)
        VALUES (?, 0, 1)
    ");
    foreach ($defaultPrizes as $amount) {
        $insert->execute([$amount]);
    }
}

// Handle delete via GET to avoid nested form issues
if (isset($_GET['action']) && $_GET['action'] === 'delete') {
    $id = (int)($_GET['id'] ?? 0);
    if ($id > 0) {
        $del = $pdo->prepare("DELETE FROM win_probabilities WHERE id = ?");
        $del->execute([$id]);
        header("Location: win_probabilities.php?success=deleted");
        exit;
    } else {
        header("Location: win_probabilities.php?error=" . urlencode('Invalid ID.'));
        exit;
    }
}

// Handle add/update via POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? 'update';

    if ($action === 'add') {
        // ADD NEW PROBABILITY ROW
        $prize_amount = (int)($_POST['prize_amount'] ?? 0);
        $percent      = (float)($_POST['percent'] ?? 0);
        $is_active    = isset($_POST['is_active']) ? 1 : 0;

        if ($prize_amount <= 0) {
            $error = 'Prize amount must be greater than 0.';
        } elseif ($percent < 0 || $percent > 100) {
            $error = 'Percent must be between 0 and 100.';
        } else {
            $stmt = $pdo->prepare("
                INSERT INTO win_probabilities (prize_amount, weight, is_active)
                VALUES (?, ?, ?)
            ");
            try {
                $stmt->execute([$prize_amount, $percent, $is_active]);
                header("Location: win_probabilities.php?success=added");
                exit;
            } catch (PDOException $e) {
                if ($e->getCode() === '23000') {
                    // Duplicate prize_amount
                    header("Location: win_probabilities.php?error=" . urlencode('Prize amount already exists.'));
                    exit;
                } else {
                    header("Location: win_probabilities.php?error=" . urlencode('Error adding record.'));
                    exit;
                }
            }
        }
    } elseif ($action === 'update') {
        // UPDATE EXISTING ROWS
        $percents = $_POST['percent'] ?? [];
        $active   = $_POST['active'] ?? [];

        try {
            $pdo->beginTransaction();

            foreach ($percents as $id => $p) {
                $id        = (int)$id;
                $percent   = (float)$p;
                if ($percent < 0)   $percent = 0;
                if ($percent > 100) $percent = 100;
                $isActive  = isset($active[$id]) ? 1 : 0;

                $update = $pdo->prepare("
                    UPDATE win_probabilities
                    SET weight = ?, is_active = ?
                    WHERE id = ?
                ");
                $update->execute([$percent, $isActive, $id]);
            }

            $pdo->commit();
            header("Location: win_probabilities.php?success=updated");
            exit;
        } catch (Exception $e) {
            $pdo->rollBack();
            header("Location: win_probabilities.php?error=" . urlencode('Error updating probabilities.'));
            exit;
        }
    }
}

// Messages from redirects
if (isset($_GET['success'])) {
    if ($_GET['success'] === 'added') {
        $success = 'New win probability added successfully.';
    } elseif ($_GET['success'] === 'updated') {
        $success = 'Win probabilities updated successfully.';
    } elseif ($_GET['success'] === 'deleted') {
        $success = 'Win probability deleted.';
    }
}
if (isset($_GET['error'])) {
    $error = $_GET['error'];
}

// Fetch probabilities
$stmt = $pdo->query("SELECT * FROM win_probabilities ORDER BY prize_amount ASC");
$rows = $stmt->fetchAll();

$totalPercent = 0;
foreach ($rows as $r) {
    if ($r['is_active']) {
        $totalPercent += (float)$r['weight']; // we treat weight as percent now
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Win Probabilities – Admin – Angpao Rain</title>
    <link rel="stylesheet"
      href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }
        body {
            font-family: Arial, sans-serif;
            background: #05070a;
            color: #f1f1f1;
            display: flex;
            min-height: 100vh;
        }

        /* ===== SIDEBAR ===== */
        .sidebar {
            width: 230px;
            background: linear-gradient(180deg, #120016, #030308);
            border-right: 1px solid rgba(255,255,255,0.08);
            padding: 18px 12px;
            display: flex;
            flex-direction: column;
        }
        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 8px 10px 18px;
            border-bottom: 1px solid rgba(255,255,255,0.08);
            margin-bottom: 10px;
        }
        .sidebar-logo-circle {
            width: 40px;
            height: 40px;
            border-radius: 999px;
            background: radial-gradient(circle, #ffeb3b, #ff9800);
            display: flex;
            align-items: center;
            justify-content: center;
            color: #4a1a00;
            font-weight: bold;
            font-size: 18px;
            box-shadow: 0 0 10px rgba(255,193,7,0.8);
        }
        .sidebar-logo-text {
            font-size: 16px;
            font-weight: bold;
            letter-spacing: 1px;
        }
        .sidebar-menu {
            margin-top: 10px;
            list-style: none;
        }
        .sidebar-menu li { margin-bottom: 4px; }
        .sidebar-link {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 9px 10px;
            border-radius: 8px;
            color: #cbd5f5;
            text-decoration: none;
            font-size: 14px;
            transition: background 0.15s, color 0.15s, transform 0.08s;
        }
        .sidebar-link i {
            width: 18px;
            text-align: center;
            font-size: 15px;
        }
        .sidebar-link:hover {
            background: rgba(255, 193, 7, 0.16);
            color: #fff9c4;
            transform: translateX(2px);
        }
        .sidebar-link.active {
            background: linear-gradient(90deg, #ff9800, #f44336);
            color: #ffffff;
            box-shadow: 0 0 12px rgba(255,111,0,0.8);
        }

        /* ===== MAIN AREA ===== */
        .main { flex: 1; display: flex; flex-direction: column; }

        .topbar {
            height: 56px;
            border-bottom: 1px solid rgba(255,255,255,0.08);
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0 20px;
            background: linear-gradient(90deg, #05060a, #12001b);
        }
        .topbar-left { font-size: 15px; color: #e5e7eb; }
        .topbar-right {
            display: flex;
            align-items: center;
            gap: 16px;
            font-size: 14px;
        }
        .topbar-user {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .topbar-avatar {
            width: 28px; height: 28px;
            border-radius: 999px;
            background: radial-gradient(circle, #ffeb3b, #ff9800);
            display: flex; align-items: center; justify-content: center;
            font-size: 15px; color: #4a1a00;
        }
        .topbar-logout {
            color:#fca5a5;
            font-size:13px;
            text-decoration:none;
        }

        .content { padding: 20px; }
        .content h1 { font-size: 22px; margin-bottom: 4px; }
        .content-subtitle {
            font-size: 13px;
            color: #9ca3af;
            margin-bottom: 18px;
        }

        .layout {
            display: grid;
            grid-template-columns: minmax(0, 260px) minmax(0, 1fr);
            gap: 18px;
            align-items: flex-start;
        }

        .card {
            background: radial-gradient(circle at top left, #1f2933, #05070a);
            border-radius: 12px;
            padding: 14px 16px;
            border: 1px solid rgba(255,255,255,0.05);
            box-shadow: 0 0 10px rgba(0,0,0,0.7);
        }

        .card-title {
            font-size: 15px;
            margin-bottom: 8px;
            font-weight: bold;
        }

        .field { margin-bottom: 10px; }
        .field label {
            display: block;
            font-size: 12px;
            margin-bottom: 4px;
            color: #d1d5db;
        }
        .input, .input-number {
            width: 100%;
            padding: 8px 9px;
            border-radius: 8px;
            border: none;
            outline: none;
            background: #111827;
            color: #f9fafb;
            font-size: 13px;
        }
        .input-number {
            width: 80px;
            padding: 5px 6px;
        }
        .input:focus, .input-number:focus {
            box-shadow: 0 0 0 1px #fbbf24;
        }

        .checkbox-row {
            display: flex;
            align-items: center;
            gap: 6px;
            font-size: 13px;
        }

        .btn {
            padding: 8px 14px;
            border-radius: 999px;
            border: none;
            outline: none;
            cursor: pointer;
            font-size: 13px;
            font-weight: bold;
            background: linear-gradient(135deg, #ff9800, #f43f5e);
            color: #fff;
            box-shadow: 0 0 10px rgba(248,113,113,0.7);
        }
        .btn:hover {
            filter: brightness(1.05);
        }

        .btn-danger-link {
            display:inline-block;
            padding: 4px 9px;
            border-radius: 999px;
            background: linear-gradient(135deg, #ef4444, #b91c1c);
            color:#fee2e2;
            font-size: 11px;
            text-decoration:none;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 13px;
        }
        th, td {
            padding: 6px 8px;
            border-bottom: 1px solid rgba(156,163,175,0.35);
            text-align: left;
        }
        th {
            background: rgba(15,23,42,0.9);
            font-weight: normal;
            color: #e5e7eb;
        }
        tr:nth-child(even) {
            background: rgba(15,23,42,0.4);
        }

        .status-msg {
            font-size: 12px;
            min-height: 16px;
            margin-bottom: 8px;
        }
        .status-success { color:#6ee7b7; }
        .status-error { color:#f87171; }

        .badge {
            display:inline-block;
            padding:2px 8px;
            border-radius:999px;
            font-size:11px;
        }
        .badge-info {
            background:rgba(59,130,246,0.2);
            color:#bfdbfe;
        }
    </style>
</head>
<body>

<!-- SIDEBAR -->
<aside class="sidebar">
    <div class="sidebar-logo">
        <div class="sidebar-logo-circle">AP</div>
        <div class="sidebar-logo-text">Angpao Admin</div>
    </div>

    <ul class="sidebar-menu">
        <li>
            <a href="index.php" class="sidebar-link">
                <i class="fas fa-gauge"></i>
                <span>Dashboard</span>
            </a>
        </li>
        <li>
            <a href="win_probabilities.php" class="sidebar-link active">
                <i class="fas fa-percentage"></i>
                <span>Win Probabilities</span>
            </a>
        </li>
        <li>
            <a href="promo_codes.php" class="sidebar-link">
                <i class="fas fa-ticket"></i>
                <span>Promo Codes</span>
            </a>
        </li>
        <li>
            <a href="promo_codes_view.php" class="sidebar-link">
                <i class="fas fa-list-ul"></i>
                <span>Promo Codes View</span>
            </a>
        </li>
        <li>
            <a href="reports.php" class="sidebar-link">
                <i class="fas fa-chart-line"></i>
                <span>Reports</span>
            </a>
        </li>
        <li>
            <a href="users.php" class="sidebar-link">
                <i class="fas fa-users"></i>
                <span>Users</span>
            </a>
        </li>
        <li>
            <a href="config_link.php" class="sidebar-link">
                <i class="fas fa-link"></i>
                <span>Config Link</span>
            </a>
        </li>
    </ul>
</aside>

<!-- MAIN AREA -->
<div class="main">
    <header class="topbar">
        <div class="topbar-left">
            Win Probabilities
        </div>
        <div class="topbar-right">
            <a href="logout.php" class="topbar-logout">
                <i class="fas fa-arrow-right-from-bracket"></i> Logout
            </a>
            <div class="topbar-user">
                <div class="topbar-avatar">A</div>
                <span><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
            </div>
        </div>
    </header>

    <main class="content">
        <h1>Configure Win Probabilities</h1>
        <div class="content-subtitle">
            Set chance (%) for each prize. You can add, edit or delete any prize row.
        </div>

        <div class="status-msg">
            <?php if ($success): ?>
                <span class="status-success"><?php echo $success; ?></span>
            <?php elseif ($error): ?>
                <span class="status-error"><?php echo htmlspecialchars($error); ?></span>
            <?php else: ?>
                <span class="status-success">
                    Total active percent: <?php echo (float)$totalPercent; ?>%
                    <?php if ($totalPercent > 0): ?>
                        <span class="badge badge-info">Used by game logic later</span>
                    <?php endif; ?>
                </span>
            <?php endif; ?>
        </div>

        <div class="layout">
            <!-- LEFT: Add row -->
            <div class="card">
                <div class="card-title">
                    <i class="fas fa-plus" style="margin-right:4px;"></i>
                    Add Prize Probability
                </div>

                <form method="post">
                    <input type="hidden" name="action" value="add">

                    <div class="field">
                        <label for="prize_amount">Prize amount (example: 1, 2, 5, 10)</label>
                        <input class="input" type="number" id="prize_amount" name="prize_amount" min="1" required>
                    </div>

                    <div class="field">
                        <label for="percent">Percent (%)</label>
                        <input class="input" type="number" id="percent" name="percent"
                               min="0" max="100" step="0.01" value="0">
                    </div>

                    <div class="field">
                        <label>Status</label>
                        <div class="checkbox-row">
                            <input type="checkbox" id="is_active" name="is_active" checked>
                            <label for="is_active" style="margin:0;">Active</label>
                        </div>
                    </div>

                    <button type="submit" class="btn">
                        <i class="fas fa-save" style="margin-right:4px;"></i>
                        Save Probability
                    </button>
                </form>
            </div>

            <!-- RIGHT: List / edit / delete -->
            <div class="card">
                <div class="card-title">
                    <i class="fas fa-sliders" style="margin-right:4px;"></i>
                    Existing Win Probabilities
                </div>

                <?php if (empty($rows)): ?>
                    <p style="font-size:13px;color:#9ca3af;margin-top:6px;">
                        No rows yet. Add at least one prize on the left.
                    </p>
                <?php else: ?>
                    <form method="post">
                        <input type="hidden" name="action" value="update">
                        <div style="overflow-x:auto;margin-top:8px;">
                            <table>
                                <thead>
                                <tr>
                                    <th>Prize</th>
                                    <th>Percent (%)</th>
                                    <th>Active</th>
                                    <th>Delete</th>
                                </tr>
                                </thead>
                                <tbody>
                                <?php foreach ($rows as $row): ?>
                                    <tr>
                                        <td>$<?php echo (int)$row['prize_amount']; ?></td>
                                        <td>
                                            <input
                                                class="input-number"
                                                type="number"
                                                name="percent[<?php echo (int)$row['id']; ?>]"
                                                value="<?php echo (float)$row['weight']; ?>"
                                                min="0" max="100" step="0.01"
                                            >
                                        </td>
                                        <td>
                                            <input
                                                type="checkbox"
                                                name="active[<?php echo (int)$row['id']; ?>]"
                                                <?php echo $row['is_active'] ? 'checked' : ''; ?>
                                            >
                                        </td>
                                        <td>
                                            <a href="win_probabilities.php?action=delete&id=<?php echo (int)$row['id']; ?>"
                                               class="btn-danger-link"
                                               onclick="return confirm('Delete this row?');">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <div style="margin-top:12px;">
                            <button type="submit" class="btn">
                                <i class="fas fa-save" style="margin-right:4px;"></i>
                                Save Probabilities
                            </button>
                        </div>
                    </form>
                <?php endif; ?>
            </div>
        </div>
    </main>
</div>

</body>
</html>
