<?php
session_start();
require __DIR__ . '/config.php';

// Protect page
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$error = '';
$success = '';

// read success message from URL (after redirect)
if (isset($_GET['success'])) {
    $success = $_GET['success'];
}

// Handle form submits
// Handle form submits (create / toggle / reset / delete)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // 1) Create new user
    if (isset($_POST['create_user'])) {
        $username   = trim($_POST['username'] ?? '');
        $password   = $_POST['password'] ?? '';
        $password2  = $_POST['password_confirm'] ?? '';
        $is_active  = isset($_POST['is_active']) ? 1 : 0;

        if ($username === '') {
            $error = 'Username is required.';
        } elseif ($password === '') {
            $error = 'Password is required.';
        } elseif (strlen($password) < 6) {
            $error = 'Password must be at least 6 characters.';
        } elseif ($password !== $password2) {
            $error = 'Password confirmation does not match.';
        } else {
            // Check unique username
            $stmt = $pdo->prepare("SELECT id FROM admin_users WHERE username = ?");
            $stmt->execute([$username]);
            if ($stmt->rowCount() > 0) {
                $error = 'This username already exists.';
            } else {
                // Insert new admin
                $hash = password_hash($password, PASSWORD_DEFAULT);

                $stmtIns = $pdo->prepare("
                    INSERT INTO admin_users (username, password_hash, is_active)
                    VALUES (?, ?, ?)
                ");

                try {
                    $stmtIns->execute([$username, $hash, $is_active]);

                    // ✅ Redirect so refresh won't re-create user
                    $msg = "Admin user '$username' created successfully.";
                    header("Location: users.php?success=" . urlencode($msg));
                    exit;

                } catch (PDOException $e) {
                    $error = 'Error creating user.';
                }
            }
        }

    // 2) Toggle Active / Inactive
    } elseif (isset($_POST['toggle_active'])) {
        $userId    = (int)($_POST['user_id'] ?? 0);
        $newStatus = (int)($_POST['new_status'] ?? 0);

        if ($userId > 0) {
            // Don't allow deactivating yourself
            if ($userId == ($_SESSION['admin_id'] ?? 0) && $newStatus == 0) {
                $error = "You cannot deactivate your own account.";
            } else {
                $stmt = $pdo->prepare("UPDATE admin_users SET is_active = ? WHERE id = ?");
                $stmt->execute([$newStatus, $userId]);

                // ✅ Redirect so refresh won't re-toggle again
                $msg = "User status updated.";
                header("Location: users.php?success=" . urlencode($msg));
                exit;
            }
        }

    // 3) Reset password
    } elseif (isset($_POST['reset_password'])) {
        $userId = (int)($_POST['user_id'] ?? 0);

        if ($userId > 0) {
            $stmt = $pdo->prepare("SELECT username FROM admin_users WHERE id = ?");
            $stmt->execute([$userId]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($user) {
                // Generate random new password
                $chars = 'ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnpqrstuvwxyz23456789';
                $newPass = '';
                for ($i = 0; $i < 10; $i++) {
                    $newPass .= $chars[random_int(0, strlen($chars) - 1)];
                }

                $hash = password_hash($newPass, PASSWORD_DEFAULT);

                $stmtUp = $pdo->prepare("UPDATE admin_users SET password_hash = ? WHERE id = ?");
                $stmtUp->execute([$hash, $userId]);

                // ✅ Redirect so refresh won't reset again
                $msg = "Password for user '" . htmlspecialchars($user['username'], ENT_QUOTES, 'UTF-8') .
                       "' has been reset. New password: " . $newPass;
                header("Location: users.php?success=" . urlencode($msg));
                exit;

            } else {
                $error = "User not found.";
            }
        }

    // 4) Delete user
    } elseif (isset($_POST['delete_user'])) {
        $userId = (int)($_POST['user_id'] ?? 0);

        if ($userId > 0) {

            // Prevent deleting yourself
            if ($userId == ($_SESSION['admin_id'] ?? 0)) {
                $error = "You cannot delete your own account.";

            } else {
                // Prevent deleting the last admin
                $stmtCnt = $pdo->query("SELECT COUNT(*) FROM admin_users");
                $totalAdmins = (int)$stmtCnt->fetchColumn();

                if ($totalAdmins <= 1) {
                    $error = "You cannot delete the last admin account.";
                } else {
                    $stmtDel = $pdo->prepare("DELETE FROM admin_users WHERE id = ?");
                    $stmtDel->execute([$userId]);

                    // ✅ Redirect so refresh won't delete again
                    $msg = "User has been deleted.";
                    header("Location: users.php?success=" . urlencode($msg));
                    exit;
                }
            }
        }
    }
}


// Fetch admin users
$stmtUsers = $pdo->query("
    SELECT id, username, is_active, created_at, last_login_at
    FROM admin_users
    ORDER BY id ASC
");
$users = $stmtUsers->fetchAll(PDO::FETCH_ASSOC);

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Users – Admin – Angpao Rain</title>
    <link rel="stylesheet"
      href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <style>
        * { box-sizing: border-box; margin: 0; padding: 0; }

        body {
            font-family: Arial, sans-serif;
            background: #05070a;
            color: #f1f1f1;
            display: flex;
            min-height: 100vh;
        }

        /* ===== SIDEBAR ===== */
        .sidebar {
            width: 230px;
            background: linear-gradient(180deg, #120016, #030308);
            border-right: 1px solid rgba(255,255,255,0.08);
            padding: 18px 12px;
            display: flex;
            flex-direction: column;
        }

        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 8px 10px 18px;
            border-bottom: 1px solid rgba(255,255,255,0.08);
            margin-bottom: 10px;
        }

        .sidebar-logo-circle {
            width: 40px;
            height: 40px;
            border-radius: 999px;
            background: radial-gradient(circle, #ffeb3b, #ff9800);
            display: flex;
            align-items: center;
            justify-content: center;
            color: #4a1a00;
            font-weight: bold;
            font-size: 18px;
            box-shadow: 0 0 10px rgba(255,193,7,0.8);
        }

        .sidebar-logo-text {
            font-size: 16px;
            font-weight: bold;
            letter-spacing: 1px;
        }

        .sidebar-menu {
            margin-top: 10px;
            list-style: none;
        }

        .sidebar-menu li {
            margin-bottom: 4px;
        }

        .sidebar-link {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 9px 10px;
            border-radius: 8px;
            color: #cbd5f5;
            text-decoration: none;
            font-size: 14px;
            transition: background 0.15s, color 0.15s, transform 0.08s;
        }

        .sidebar-link i {
            width: 18px;
            text-align: center;
            font-size: 15px;
        }

        .sidebar-link:hover {
            background: rgba(255, 193, 7, 0.16);
            color: #fff9c4;
            transform: translateX(2px);
        }

        .sidebar-link.active {
            background: linear-gradient(90deg, #ff9800, #f44336);
            color: #ffffff;
            box-shadow: 0 0 12px rgba(255,111,0,0.8);
        }

        /* ===== MAIN ===== */
        .main {
            flex: 1;
            display: flex;
            flex-direction: column;
        }

        .topbar {
            height: 56px;
            border-bottom: 1px solid rgba(255,255,255,0.08);
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0 20px;
            background: linear-gradient(90deg, #05060a, #12001b);
        }

        .topbar-left { font-size: 15px; color: #e5e7eb; }
        .topbar-right { display:flex;align-items:center;gap:16px;font-size:14px; }
        .topbar-logout { color:#fca5a5;font-size:13px;text-decoration:none; }
        .topbar-user { display:flex;align-items:center;gap:8px; }

        .topbar-avatar {
            width: 28px;
            height: 28px;
            border-radius: 999px;
            background: radial-gradient(circle, #ffeb3b, #ff9800);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 15px;
            color: #4a1a00;
        }

        .content { padding: 20px; }
        .content h1 { font-size: 22px; margin-bottom: 4px; }
        .content-subtitle { font-size: 13px; color:#9ca3af; margin-bottom:18px; }

        .grid-2 {
            display:grid;
            grid-template-columns:minmax(0,280px) minmax(0,1fr);
            gap:18px;
        }

        .card {
            background:radial-gradient(circle at top left,#1f2933,#05070a);
            border-radius:12px;
            padding:14px 16px;
            border:1px solid rgba(255,255,255,0.05);
            box-shadow:0 0 10px rgba(0,0,0,0.7);
        }

        .card-title {
            font-size:14px;
            margin-bottom:10px;
            font-weight:bold;
        }

        .field { margin-bottom:10px; }
        .field label {
            display:block;
            font-size:12px;
            margin-bottom:4px;
            color:#d1d5db;
        }

        .input {
            width:100%;
            padding:8px 9px;
            border-radius:8px;
            border:none;
            outline:none;
            background:#111827;
            color:#f9fafb;
            font-size:13px;
        }

        .input:focus { box-shadow:0 0 0 1px #fbbf24; }

        .checkbox-row {
            display:flex;
            align-items:center;
            gap:6px;
            font-size:13px;
        }

        .btn {
            padding:8px 14px;
            border-radius:999px;
            border:none;
            outline:none;
            cursor:pointer;
            font-size:13px;
            font-weight:bold;
            background:linear-gradient(135deg,#ff9800,#f43f5e);
            color:#fff;
            box-shadow:0 0 10px rgba(248,113,113,0.7);
        }

        .status-msg {
            font-size:12px;
            min-height:16px;
            margin-bottom:6px;
        }

        .status-success { color:#6ee7b7; }
        .status-error   { color:#f87171; }

        table {
            width:100%;
            border-collapse:collapse;
            font-size:12px;
        }

        th, td {
            padding:6px 8px;
            border-bottom:1px solid rgba(156,163,175,0.35);
            text-align:left;
        }

        th {
            background:rgba(15,23,42,0.9);
            font-weight:normal;
            color:#e5e7eb;
        }

        tr:nth-child(even) { background:rgba(15,23,42,0.5); }

        .badge-active {
            display:inline-block;
            padding:2px 8px;
            border-radius:999px;
            background:rgba(16,185,129,0.2);
            color:#6ee7b7;
            font-size:11px;
        }

        .badge-inactive {
            display:inline-block;
            padding:2px 8px;
            border-radius:999px;
            background:rgba(239,68,68,0.2);
            color:#fecaca;
            font-size:11px;
        }
    </style>
</head>
<body>

<!-- SIDEBAR -->
<aside class="sidebar">
    <div class="sidebar-logo">
        <div class="sidebar-logo-circle">AP</div>
        <div class="sidebar-logo-text">Angpao Admin</div>
    </div>

    <ul class="sidebar-menu">
        <li><a href="index.php" class="sidebar-link"><i class="fas fa-gauge"></i><span>Dashboard</span></a></li>
        <li><a href="win_probabilities.php" class="sidebar-link"><i class="fas fa-percentage"></i><span>Win Probabilities</span></a></li>
        <li><a href="promo_codes.php" class="sidebar-link"><i class="fas fa-ticket"></i><span>Promo Codes</span></a></li>
        <li><a href="promo_codes_view.php" class="sidebar-link"><i class="fas fa-list-ul"></i><span>Promo Codes View</span></a></li>
        <li><a href="reports.php" class="sidebar-link"><i class="fas fa-chart-line"></i><span>Reports</span></a></li>
        <li><a href="users.php" class="sidebar-link active"><i class="fas fa-users"></i><span>Users</span></a></li>
        <li><a href="config_link.php" class="sidebar-link"><i class="fas fa-link"></i><span>Config Link</span></a></li>
    </ul>
</aside>

<!-- MAIN AREA -->
<div class="main">
    <header class="topbar">
        <div class="topbar-left">Users</div>
        <div class="topbar-right">
            <a href="logout.php" class="topbar-logout">
                <i class="fas fa-arrow-right-from-bracket"></i> Logout
            </a>
            <div class="topbar-user">
                <div class="topbar-avatar">A</div>
                <span><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
            </div>
        </div>
    </header>

    <main class="content">
        <h1>Admin Users</h1>
        <div class="content-subtitle">
            Create and manage admin accounts that can access this control panel.
        </div>

        <div class="grid-2">
            <!-- Left: create user -->
            <div class="card">
                <div class="card-title">
                    <i class="fas fa-user-plus" style="margin-right:4px;"></i> New Admin User
                </div>

                <div class="status-msg">
                    <?php if ($success): ?>
                        <span class="status-success"><?php echo $success; ?></span>
                    <?php elseif ($error): ?>
                        <span class="status-error"><?php echo $error; ?></span>
                    <?php endif; ?>
                </div>

                <form method="post">
                    <div class="field">
                        <label for="username">Username</label>
                        <input class="input" type="text" id="username" name="username" required>
                    </div>

                    <div class="field">
                        <label for="password">Password</label>
                        <input class="input" type="password" id="password" name="password" required>
                    </div>

                    <div class="field">
                        <label for="password_confirm">Confirm Password</label>
                        <input class="input" type="password" id="password_confirm" name="password_confirm" required>
                    </div>

                    <div class="field">
                        <label>Status</label>
                        <div class="checkbox-row">
                            <input type="checkbox" id="is_active" name="is_active" checked>
                            <label for="is_active" style="margin:0;">Active</label>
                        </div>
                    </div>

                    <button type="submit" class="btn" name="create_user" value="1">
                        <i class="fas fa-save" style="margin-right:4px;"></i> Save User
                    </button>
                </form>
            </div>

            <!-- Right: user list -->
            <div class="card">
                <div class="card-title">
                    <i class="fas fa-users" style="margin-right:4px;"></i> Existing Admins
                </div>

                <?php if (empty($users)): ?>
                    <p style="font-size:13px;color:#9ca3af;margin-top:6px;">
                        No admin users created yet.
                    </p>
                <?php else: ?>
                    <div style="overflow-x:auto;margin-top:6px;">
                        <table>
                            <thead>
                            <tr>
                                <th>ID</th>
                                <th>Username</th>
                                <th>Status</th>
                                <th>Created</th>
                                <th>Last Login</th>
                                <th>Actions</th>
                            </tr>
                            </thead>
                            <tbody>
                            <?php foreach ($users as $row): ?>
                                <tr>
                                    <td><?php echo (int)$row['id']; ?></td>
                                    <td><?php echo htmlspecialchars($row['username']); ?></td>
                                    <td>
                                        <?php if ($row['is_active']): ?>
                                            <span class="badge-active">Active</span>
                                        <?php else: ?>
                                            <span class="badge-inactive">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($row['created_at']); ?></td>
                                    <td><?php echo htmlspecialchars($row['last_login_at'] ?? '-'); ?></td>
                                    <td>
                                        <!-- Toggle Active / Inactive -->
                                        <?php if ($row['id'] != ($_SESSION['admin_id'] ?? 0)): ?>
                                            <form method="post" style="display:inline;">
                                                <input type="hidden" name="user_id" value="<?php echo (int)$row['id']; ?>">
                                                <input type="hidden" name="new_status" value="<?php echo $row['is_active'] ? 0 : 1; ?>">
                                                <button type="submit" name="toggle_active" value="1"
                                                        style="border:none;border-radius:999px;padding:4px 10px;font-size:11px;cursor:pointer;
                                                               background:<?php echo $row['is_active'] ? '#b91c1c' : '#16a34a'; ?>;
                                                               color:#fff;">
                                                    <?php echo $row['is_active'] ? 'Deactivate' : 'Activate'; ?>
                                                </button>
                                            </form>
                                        <?php else: ?>
                                            <span style="font-size:11px;color:#9ca3af;">(You)</span>
                                        <?php endif; ?>

                                        <!-- Reset password -->
                                        <form method="post" style="display:inline;margin-left:6px;">
                                            <input type="hidden" name="user_id" value="<?php echo (int)$row['id']; ?>">
                                            <button type="submit" name="reset_password" value="1"
                                                    style="border:none;border-radius:999px;padding:4px 10px;font-size:11px;cursor:pointer;
                                                           background:#0369a1;color:#fff;">
                                                Reset Pass
                                            </button>
                                        </form>

                                        <!-- Delete user -->
                                        <?php if ($row['id'] != ($_SESSION['admin_id'] ?? 0)): ?>
                                            <form method="post" style="display:inline;margin-left:6px;"
                                                  onsubmit="return confirm('Are you sure you want to delete this user?');">
                                                <input type="hidden" name="user_id" value="<?php echo (int)$row['id']; ?>">
                                                <button type="submit" name="delete_user" value="1"
                                                        style="border:none;border-radius:999px;padding:4px 10px;font-size:11px;cursor:pointer;
                                                               background:#991b1b;color:#fee2e2;">
                                                    Delete
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </main>
</div>

</body>
</html>
