<?php
session_start();
require __DIR__ . '/config.php';

// Protect page
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$error = '';
$success = '';
if (isset($_GET['success'])) {
    $success = "Promo code: " . htmlspecialchars($_GET['success']) . " created successfully.";
}

// Handle new promo form submit
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $code = strtoupper(trim($_POST['code'] ?? ''));

    // Auto-generate if blank
    if ($code === '') {
        $code = 'PROMO-' . strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 8));
    }

    // Ensure unique
    while (true) {
        $stmtCheck = $pdo->prepare("SELECT id FROM promo_codes WHERE code = ?");
        $stmtCheck->execute([$code]);

        if ($stmtCheck->rowCount() == 0) break;

        // generate new if collision
        $code = 'PROMO-' . strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 8));
    }

    $description = trim($_POST['description'] ?? '');
    $max_uses    = (int)($_POST['max_uses'] ?? 1);
    $is_active   = isset($_POST['is_active']) ? 1 : 0;

    if ($code === '') {
        $error = 'Promo code is required.';
    } elseif ($max_uses < 1) {
        $error = 'Max uses must be at least 1.';
    } else {
        try {
            // ✅ CORRECT INSERT: save code, description, max_uses, is_active
            $stmt = $pdo->prepare("
                INSERT INTO promo_codes (code, description, max_uses, is_active)
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute([$code, $description, $max_uses, $is_active]);

            // Redirect with success message (PRG pattern)
            header("Location: promo_codes.php?success=" . urlencode($code));
            exit;

        } catch (PDOException $e) {
            if ($e->getCode() === '23000') {
                $error = 'This promo code already exists.';
            } else {
                $error = 'Error saving promo code.';
            }
        }
    }
}


// Fetch promo codes
$stmt = $pdo->query("SELECT * FROM promo_codes ORDER BY created_at DESC");
$promoCodes = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Promo Codes – Admin – Angpao Rain</title>
    <link rel="stylesheet"
      href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: Arial, sans-serif;
            background: #05070a;
            color: #f1f1f1;
            display: flex;
            min-height: 100vh;
        }

        /* ===== SIDEBAR ===== */
        .sidebar {
            width: 230px;
            background: linear-gradient(180deg, #120016, #030308);
            border-right: 1px solid rgba(255,255,255,0.08);
            padding: 18px 12px;
            display: flex;
            flex-direction: column;
        }

        .sidebar-logo {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 8px 10px 18px;
            border-bottom: 1px solid rgba(255,255,255,0.08);
            margin-bottom: 10px;
        }

        .sidebar-logo-circle {
            width: 40px;
            height: 40px;
            border-radius: 999px;
            background: radial-gradient(circle, #ffeb3b, #ff9800);
            display: flex;
            align-items: center;
            justify-content: center;
            color: #4a1a00;
            font-weight: bold;
            font-size: 18px;
            box-shadow: 0 0 10px rgba(255,193,7,0.8);
        }

        .sidebar-logo-text {
            font-size: 16px;
            font-weight: bold;
            letter-spacing: 1px;
        }

        .sidebar-menu {
            margin-top: 10px;
            list-style: none;
        }

        .sidebar-menu li {
            margin-bottom: 4px;
        }

        .sidebar-link {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 9px 10px;
            border-radius: 8px;
            color: #cbd5f5;
            text-decoration: none;
            font-size: 14px;
            transition: background 0.15s, color 0.15s, transform 0.08s;
        }

        .sidebar-link i {
            width: 18px;
            text-align: center;
            font-size: 15px;
        }

        .sidebar-link:hover {
            background: rgba(255, 193, 7, 0.16);
            color: #fff9c4;
            transform: translateX(2px);
        }

        .sidebar-link.active {
            background: linear-gradient(90deg, #ff9800, #f44336);
            color: #ffffff;
            box-shadow: 0 0 12px rgba(255,111,0,0.8);
        }

        /* ===== MAIN CONTENT AREA ===== */
        .main {
            flex: 1;
            display: flex;
            flex-direction: column;
        }

        /* Top bar */
        .topbar {
            height: 56px;
            border-bottom: 1px solid rgba(255,255,255,0.08);
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0 20px;
            background: linear-gradient(90deg, #05060a, #12001b);
        }

        .topbar-left {
            font-size: 15px;
            color: #e5e7eb;
        }

        .topbar-right {
            display: flex;
            align-items: center;
            gap: 16px;
            font-size: 14px;
        }

        .topbar-right i {
            opacity: 0.8;
        }

        .topbar-user {
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .topbar-avatar {
            width: 28px;
            height: 28px;
            border-radius: 999px;
            background: radial-gradient(circle, #ffeb3b, #ff9800);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 15px;
            color: #4a1a00;
        }

        .topbar-logout {
            color:#fca5a5;
            font-size:13px;
            text-decoration:none;
        }

        /* Content area */
        .content {
            padding: 20px;
        }

        .content h1 {
            font-size: 22px;
            margin-bottom: 4px;
        }

        .content-subtitle {
            font-size: 13px;
            color: #9ca3af;
            margin-bottom: 18px;
        }

        .grid-2 {
            display: grid;
            grid-template-columns: minmax(0, 280px) minmax(0, 1fr);
            gap: 18px;
        }

        .card {
            background: radial-gradient(circle at top left, #1f2933, #05070a);
            border-radius: 12px;
            padding: 14px 16px;
            border: 1px solid rgba(255,255,255,0.05);
            box-shadow: 0 0 10px rgba(0,0,0,0.7);
        }

        .card-title {
            font-size: 14px;
            margin-bottom: 10px;
            font-weight: bold;
        }

        .field {
            margin-bottom: 10px;
        }

        .field label {
            display: block;
            font-size: 12px;
            margin-bottom: 4px;
            color: #d1d5db;
        }

        .input, .textarea {
            width: 100%;
            padding: 8px 9px;
            border-radius: 8px;
            border: none;
            outline: none;
            background: #111827;
            color: #f9fafb;
            font-size: 13px;
        }

        .input:focus, .textarea:focus {
            box-shadow: 0 0 0 1px #fbbf24;
        }

        .textarea {
            resize: vertical;
            min-height: 60px;
        }

        .checkbox-row {
            display: flex;
            align-items: center;
            gap: 6px;
            font-size: 13px;
        }

        .btn {
            padding: 8px 14px;
            border-radius: 999px;
            border: none;
            outline: none;
            cursor: pointer;
            font-size: 13px;
            font-weight: bold;
            background: linear-gradient(135deg, #ff9800, #f43f5e);
            color: #fff;
            box-shadow: 0 0 10px rgba(248,113,113,0.7);
        }

        .status-msg {
            font-size: 12px;
            min-height: 16px;
            margin-bottom: 6px;
        }

        .status-success {
            color: #6ee7b7;
        }

        .status-error {
            color: #f87171;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 12px;
        }

        th, td {
            padding: 6px 8px;
            border-bottom: 1px solid rgba(156,163,175,0.35);
            text-align: left;
        }

        th {
            background: rgba(15,23,42,0.9);
            font-weight: normal;
            color: #e5e7eb;
        }

        tr:nth-child(even) {
            background: rgba(15,23,42,0.5);
        }

        .badge-active {
            display:inline-block;
            padding:2px 8px;
            border-radius:999px;
            background:rgba(16,185,129,0.2);
            color:#6ee7b7;
            font-size:11px;
        }

        .badge-inactive {
            display:inline-block;
            padding:2px 8px;
            border-radius:999px;
            background:rgba(239,68,68,0.2);
            color:#fecaca;
            font-size:11px;
        }
    </style>
</head>
<body>

<!-- SIDEBAR -->
<aside class="sidebar">
    <div class="sidebar-logo">
        <div class="sidebar-logo-circle">
            AP
        </div>
        <div class="sidebar-logo-text">
            Angpao Admin
        </div>
    </div>

    <ul class="sidebar-menu">
        <li>
            <a href="index.php" class="sidebar-link">
                <i class="fas fa-gauge"></i>
                <span>Dashboard</span>
            </a>
        </li>
        <li>
            <a href="win_probabilities.php" class="sidebar-link">
                <i class="fas fa-percentage"></i>
                <span>Win Probabilities</span>
            </a>
        </li>
        <li>
            <a href="promo_codes.php" class="sidebar-link active">
                <i class="fas fa-ticket"></i>
                <span>Promo Codes</span>
            </a>
        </li>
        <li>
            <a href="promo_codes_view.php" class="sidebar-link">
                <i class="fas fa-list-ul"></i>
                <span>Promo Codes View</span>
            </a>
        </li>
        <li>
            <a href="reports.php" class="sidebar-link">
                <i class="fas fa-chart-line"></i>
                <span>Reports</span>
            </a>
        </li>
        <li>
            <a href="users.php" class="sidebar-link">
                <i class="fas fa-users"></i>
                <span>Users</span>
            </a>
        </li>
        <li>
            <a href="config_link.php" class="sidebar-link">
                <i class="fas fa-link"></i>
                <span>Config Link</span>
            </a>
        </li>
    </ul>
</aside>

<!-- MAIN AREA -->
<div class="main">
    <!-- Topbar -->
    <header class="topbar">
        <div class="topbar-left">
            Promo Codes
        </div>
        <div class="topbar-right">
            <a href="logout.php" class="topbar-logout">
                <i class="fas fa-arrow-right-from-bracket"></i> Logout
            </a>
            <div class="topbar-user">
                <div class="topbar-avatar">A</div>
                <span><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
            </div>
        </div>
    </header>

    <!-- Content -->
    <main class="content">
        <h1>Manage Promo Codes</h1>
        <div class="content-subtitle">
            Create and manage promocodes that can be used in the Angpao Rain game.
        </div>

        <div class="grid-2">
            <!-- Left: create form -->
            <div class="card">
                <div class="card-title">
                    <i class="fas fa-plus" style="margin-right:4px;"></i> New Promo Code
                </div>

                <div class="status-msg">
                    <?php if ($success): ?>
                        <span class="status-success"><?php echo htmlspecialchars($success); ?></span>
                    <?php elseif ($error): ?>
                        <span class="status-error"><?php echo htmlspecialchars($error); ?></span>
                    <?php endif; ?>
                </div>

                <form method="post">
                    <div class="field">
                        <label for="code">Code (example: PROMO-UCVSNUPZ)</label>
                        <input class="input" type="text" id="code" name="code">
                    </div>

                    <div class="field">
                        <label for="description">Description (optional)</label>
                        <textarea class="textarea" id="description" name="description"
                                  placeholder="Event name, campaign, note..."></textarea>
                    </div>

                    <div class="field">
                        <label for="max_uses">Max Uses</label>
                        <input class="input" type="number" id="max_uses" name="max_uses" value="1" min="1">
                    </div>

                    <div class="field">
                        <label>Status</label>
                        <div class="checkbox-row">
                            <input type="checkbox" id="is_active" name="is_active" checked>
                            <label for="is_active" style="margin:0;">Active</label>
                        </div>
                    </div>

                    <button type="submit" class="btn">
                        <i class="fas fa-save" style="margin-right:4px;"></i> Save Promo Code
                    </button>
                </form>
            </div>

            <!-- Right: table list -->
            <div class="card">
                <div class="card-title">
                    <i class="fas fa-list-ul" style="margin-right:4px;"></i> Existing Promo Codes
                </div>

                <?php if (empty($promoCodes)): ?>
                    <p style="font-size:13px;color:#9ca3af;margin-top:6px;">
                        No promo codes created yet.
                    </p>
                <?php else: ?>
                    <div style="overflow-x:auto;margin-top:6px;">
                        <table>
                            <thead>
                            <tr>
                                <th>ID</th>
                                <th>Code</th>
                                <th>Description</th>
                                <th>Uses</th>
                                <th>Status</th>
                                <th>Created</th>
                            </tr>
                            </thead>
                            <tbody>
                            <?php foreach ($promoCodes as $row): ?>
                                <tr>
                                    <td><?php echo (int)$row['id']; ?></td>
                                    <td><?php echo htmlspecialchars($row['code']); ?></td>
                                    <td><?php echo htmlspecialchars($row['description'] ?? ''); ?></td>
                                    <td>
                                        <?php echo (int)$row['used_count']; ?>
                                        /
                                        <?php echo (int)$row['max_uses']; ?>
                                    </td>
                                    <td>
                                        <?php if ($row['is_active']): ?>
                                            <span class="badge-active">Active</span>
                                        <?php else: ?>
                                            <span class="badge-inactive">Inactive</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($row['created_at']); ?></td>
                                </tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </main>
</div>

</body>
</html>
